;=================================================
;  *Copyright(C) 2023-2024 Advanced Micro Devices, Inc. All rights Reserved.
;=================================================

; Camera sensor I2C group control API
;
; [void] SensorGroupHold(void)
; [void] SensorGroupRelease(void)
;=================================================
.function SensorGroupHold
WI2C $0x3208, $0x00, $1 ;Group Hold
.endfunction

.function SensorGroupRelease
WI2C $0x3208, $0x10, $1 ;Group Release
WI2C $0x320D, $0x00, $1  
WI2C $0x3208, $0xA0, $1  
.endfunction
;=================================================
; Camera sensor exposure control API
;
; [uint32 setGain]  SensorConvertAGain(uint32 aGain, uint32 *aGainReal, uint32 *aGainSetting)
;=================================================
.function SensorConvertAGain
;R0 is the aGain
;gain_reg_coarse = aGain / 1000;
;gain_reg_fine = (aGain*10-(gain_reg_coarse*10000)) / 625;
;if(gain_reg_coarse<2): gain_reg_fine
;elseif(gain_reg_coarse<4):gain_reg_fine = gain_reg_fine&0xe
;else:gain_reg_fine = gain_reg_fine&0xc
;R1 = gain_reg_coarse << 8 | gain_reg_fine << 4
;R0 = gain_reg_coarse*1000+gain_reg_fine*625/10
;return R0&R1
START:
;calculate gain_reg_coarse
DIV R8, R0, $1000           ;R8 = R0/1000;	gain_reg_coarse

;calculate gain_reg_fine
MUL R9, R8, $10000          ;R9 = R8 * 10000
MUL R10, R0, $10            ;R10 = R0 * 10
SUB R11, R10,R9				;R11 = R10 - R9
DIV R12, R11, $625          ;R12 = R11/625;	gain_reg_fine

;dispatch 
DIV R13,R8,$2				;R13 = R8 / 2
JEZ R13,@FL_AGAIN2			;if (R8<2),jump to FL_AGAIN2
DIV R13,R8,$4				;R13 = R8 / 4
JEZ R13,@FL_AGAIN4			;if (2<=R8<4),jump to FL_AGAIN4
DIV R13,R8,$8				;R13 = R8 / 8
JEZ R13,@FL_AGAIN8			;if (4<=R8<8),jump to FL_AGAIN8
DIV R13,R8,$16				;R13 = R8 / 16
JEZ R13,@FL_AGAIN16			;if (8<=R8<16),jump to FL_AGAIN16
MOVDW R0,$15500				;Exception->set Again = 15.5 max
JEZ $0,@START				;RE calculate again = 15.5*1000

FL_AGAIN2:
JEZ $0,@END					;

FL_AGAIN4:
AND R12,R12,$0xe			;
JEZ $0,@END					;

FL_AGAIN8:
AND R12,R12,$0xc			;
JEZ $0,@END					;

FL_AGAIN16:
AND R12,R12,$0x8			;
JEZ $0,@END					;

END:
;create aGainSetting
SHL R13,R8,$8				;R13 = R8 << 8  
SHL R14,R12,$4				;R14 = R12 << 4 
OR R1,R13,R14				;R1 = R13 | R14

;create aGainReal
MUL R15,R8,$1000			;R15 = R8 * 1000
MUL R16,R12,$625			;R16 = R12 * 625
DIV R17,R16,$10				;R17 = R16 / 10	;may lose 0.0005 AGain
ADD R0, R15,R17				;R0 = R15 + R17


.endfunction

;=================================================
; Camera sensor exposure control API
;
; [uint32 setGain]  SensorSetAGain(uint32 aGainSetting)
;=================================================
;R0 is the aGainSetting
.function SensorSetAGain
SHR R10, R0, $8            	;R10 = aGainSetting >> 8
AND R10, R10,$0xff			;
AND R11,R0,$0xff			;R11 = R0 & 0xff
SHL R11,R11,$8				;R11 = R11 << 8
OR  R11,R11,R10				;R11 = R11 | R10

WI2C $0x3508, R11, $2

.endfunction

;=================================================
; Camera sensor exposure control API
;
; [uint32 setGain]  SensorSetDGain(uint32 dGain)
;=================================================

.function SensorSetDGain
;Digital_gain = {R350a[3:0] , R350b[7:0] , R350c[7:6]} / 1024
;R0 is gain

SHL R0, R0, $10
DIV R4, R0, $1000 
AND R5, R4, $0x000003  ;R5(low 2bit)
SHL R5, R5, $6
AND R6, R4, $0x003ff    ;R6(mid 8bit)
SHR R6, R6, $2
SHR R7, R4, $10

WI2C $0x350a, R7, $1
WI2C $0x350b, R6, $1
WI2C $0x350c, R5, $1
.endfunction

;=================================================
; Camera sensor exposure control API
;
; [uint32 getGain]  SensorGetDGain(uint32 dGainSetting)
;=================================================

.function SensorGetDGain
;Digital_gain = {R350a[3:0] , R350b[7:0] , R350c[7:6]} / 1024
;R0 is dgain(*1000)

RI2C R8,  $0x350a, $1
RI2C R9,  $0x350b, $1
RI2C R10, $0x350c, $1

AND R8, R8, $0x0f
AND R10, R10, $0xc0
SHL R8, R8, $10         ;R8 = R8 << (2+8)
SHL R9, R9, $2          ;R9 = R8 << 2
SHR R11, R10, $6        ;R11 = R10 >> 6
OR R11, R9, R11
OR R11, R8, R11         ;R11 = R8||R9||R11

MUL R0, R11, $1000      ;R11 = R11*1000
SHR R0, R0, $10         ;R11 = R11 >> 10 = R11/1024
.endfunction

;=================================================
; Camera sensor exposure control API
;[uint32 getAGain] SensorGetAGain(uint32 realGain)
;=================================================
.function SensorGetAGain
;R0 is the realGain
;Gain = (1000 * regGain) / 256;
RI2C R8, $0x3508, $1   ;
RI2C R9, $0x3509, $1   ;
SHL R10, R8, $8        ;
OR R10, R10, R9        ;

MUL R10, R10, $1000    ;
DIV R0, R10, $256      ;
.endfunction

;=================================================
; Camera sensor exposure control API
;
; [uint32 setItime] SensorSetItime(uint32 iTimeSetting,
;                                  uint32 durationSetting)
;=================================================
.function SensorSetItime
;R0 is the iTimeSetting    exposure  unit:line
;R1 is the durationSetting vts
;R6: first_frame_length_lines
;R31: save the current_frame_length_lines

JNZ R31 @FL_UPDATED
MOVDW R31, R6               ;R31 = R6;
FL_UPDATED:
SUB R30, R1, R31            ;R30 = R1 - R31 = durationSetting(frame_length_line) - current_frame_length_lines
MOVDW R31, R1               ;R31 = R1

AND R10, R1, $0xff        	;R10 =  frame_length_line & 0xff, low 8 bit
SHR R11, R1, $8				;R11 = (frame_length_line>>8) & 0Xff, high 8 bit
AND R11, R11, $0xff			;

AND R15, R0, $0x0000ff		;R15 (low 8bit)
SHL R15, R15,$16			;R15<<16
AND R16, R0, $0x00ff00		;R16(mid 8bit)
AND R17, R0, $0xff0000		;R17(high 8bit)
SHR R17, R17,$16			;R17>>16 
OR R18, R15,R16      		;R18 = R15 | R16 | R17
OR R18, R18,R17      		;

;JLZ R30 @LINE_COUNT_FIRST  ;durationSetting is smaller than current_frame_length_lines
WI2C $0x380e, R11, $1
WI2C $0x380f, R10, $1
WI2C $0x3500, R18, $3

JGZ R6 @END                ;R6 should bigger than 0 always, directly jump to end

LINE_COUNT_FIRST:
WI2C $0x3500, R18, $3
WI2C $0x380e, R11, $1
WI2C $0x380f, R10, $1

END:
;Note, the setted new Gain and new Itime must be return in R0 and R1
;Here, we just return the input value.
MOVDW R0, R0
.endfunction

;=================================================
; Camera sensor exposure control API
;
;=================================================
.function SensorGetItime 
;R0 is the newItime
;R1: t_line

RI2C R9, $0x3500, $1   ;line_count_h
RI2C R10, $0x3501, $1  ;line_count_m
RI2C R11, $0x3502, $1  ;line_count_l
SHL R12, R9, $16       ;R12 = R9(line_count_h) << 16
SHL R13, R10, $8       ;R13 = R10(line_count_m) << 8
OR R14, R11, R12       ;R14 = R12 | R13 | R11
OR R14, R14, R13       ;
MUL R0, R14, $1        ;R0 = R14 

.endfunction

;=============================================================================
;                    For HDR High Analog Gain
;
;    [uint32 setGain]  SensorSetHdrHighAGain(uint32 aGainSetting)
;=============================================================================
.function SensorSetHdrHighAGain
CALL @SensorSetAGain
.endfunction

;=============================================================================
;                    For HDR Low Analog Gain
;
;    [uint32 setGain]  SensorSetHdrLowAGain(uint32 aGainSetting)
;=============================================================================
.function SensorSetHdrLowAGain
;R0 is the aGainSetting

SHR R10, R0, $8             ;R10 = aGainSetting >> 8
AND R10, R10,$0xff          ;
AND R11,R0,$0xff            ;R11 = R0 & 0xff
SHL R11,R11,$8              ;R11 = R11 << 8
OR  R11,R11,R10             ;R11 = R11 | R10

WI2C $0x3548, R11, $2

.endfunction

;=============================================================================
;                    For HDR High Digital Gain
;
;    [uint32 setGain]  SensorSetHdrHighDGain(uint32 dGain)
;=============================================================================
.function SensorSetHdrHighDGain
CALL @SensorSetDGain
.endfunction

;=============================================================================
;                    For HDR Low Digital Gain
;
;    [uint32 setGain]  SensorSetHdrLowDGain(uint32 dGain)
;=============================================================================
.function SensorSetHdrLowDGain
;R0 is the dGain
;Digital_gain = {R354a[3:0] , R354b[7:0] , R354c[7:6]} / 1024

SHL R0, R0, $10
DIV R4, R0, $1000 
AND R5, R4, $0x000003  ;R5(low 2bit)
SHL R5, R5, $6
AND R6, R4, $0x003ff    ;R6(mid 8bit)
SHR R6, R6, $2   
SHR R7, R4, $10

WI2C $0x354a, R7, $1
WI2C $0x354b, R6, $1
WI2C $0x354c, R5, $1

.endfunction

;=============================================================================
;                    For HDR High exposure control API
;
;    [uint32 setItime] SensorSetHdrHighItime(uint32 iTimeSetting)
;=============================================================================
.function SensorSetHdrHighItime
;R0 is the iTimeSetting    exposure  unit:line

AND R15, R0,  $0x0000ff      ;R15 (low 8bit)
SHL R15, R15, $16            ;R15<<16
AND R16, R0,  $0x00ff00      ;R16(mid 8bit)
AND R17, R0,  $0xff0000      ;R17(high 8bit)
SHR R17, R17, $16            ;R17>>16 
OR R18, R15,  R16            ;R18 = R15 | R16 | R17
OR R18, R18,  R17            ;

WI2C $0x3500, R18, $3

END:
;Note, the setted new Gain and new Itime must be return in R0 and R1
;Here, we just return the input value.
MOVDW R0, R0
.endfunction

;=================================================
; Camera sensor exposure control API
;
; [uint32 setItime] SensorSetHdrLowItime(uint32 iTimeSetting)
;=================================================
.function SensorSetHdrLowItime
;R0 is the iTimeSetting    exposure  unit:line

AND R15, R0,  $0x0000ff      ;R15 (low 8bit)
SHL R15, R15, $16            ;R15<<16
AND R16, R0,  $0x00ff00      ;R16(mid 8bit)
AND R17, R0,  $0xff0000      ;R17(high 8bit)
SHR R17, R17, $16            ;R17>>16 
OR R18, R15,  R16            ;R18 = R15 | R16 | R17
OR R18, R18,  R17            ;

WI2C $0x3540, R18, $3

END:
;Note, the setted new Gain and new Itime must be return in R0 and R1
;Here, we just return the input value.
MOVDW R0, R0
.endfunction

.function SensorStreamOn
MOVDW R4, $1      ; R4 = 1
WI2C $0x0100, R4, $1
.endfunction

;=================================================
; Camera sensor exposure control API
;
; SensorSetDuration(uint32 durationSetting)
;=================================================
.function SensorSetDuration
;R0 is the durationSetting vts
AND R10, R0, $0xff          ;R10 =  frame_length_line & 0xff, low 8 bit
SHR R11, R0, $8             ;R11 = (frame_length_line>>8) & 0Xff, high 8 bit
AND R11, R11, $0xff         ;

WI2C $0x380e, R11, $1
WI2C $0x380f, R10, $1

.endfunction

;================================================= 
; Camera sensor exposure control API ; 
; SensorGetDuration(uint32 pdurationSetting) 
;================================================= 
.function SensorGetDuration 
;R0 is the durationSetting vts

RI2C R9, $0x380e, $1   ;vts_h
RI2C R10, $0x380f, $1  ;vts_l
SHL R11, R9, $8        ;R11 = R9(vts_h) << 8
OR R12, R11, R10       ;R14 = R11 | R10
MUL R0, R12, $1        ;R0 = R12
.endfunction
