/*******************************************************************************
 * Copyright (c) PLX Technology, Inc.
 *
 * PLX Technology Inc. licenses this source file under the GNU Lesser General Public
 * License (LGPL) version 2.  This source file may be modified or redistributed
 * under the terms of the LGPL and without express permission from PLX Technology.
 *
 * PLX Technology, Inc. provides this software AS IS, WITHOUT ANY WARRANTY,
 * EXPRESS OR IMPLIED, INCLUDING, WITHOUT LIMITATION, ANY WARRANTY OF
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  PLX makes no guarantee
 * or representations regarding the use of, or the results of the use of,
 * the software and documentation in terms of correctness, accuracy,
 * reliability, currentness, or otherwise; and you rely on the software,
 * documentation and results solely at your own risk.
 *
 * IN NO EVENT SHALL PLX BE LIABLE FOR ANY LOSS OF USE, LOSS OF BUSINESS,
 * LOSS OF PROFITS, INDIRECT, INCIDENTAL, SPECIAL OR CONSEQUENTIAL DAMAGES
 * OF ANY KIND.
 *
 ******************************************************************************/

/******************************************************************************
 *
 * File Name:
 *
 *      PlxChipFn.c
 *
 * Description:
 *
 *      Contains PLX chip-specific support functions
 *
 * Revision History:
 *
 *      09-01-10 : PLX SDK v6.40
 *
 ******************************************************************************/


#include "ApiFunc.h"
#include "PciFunc.h"
#include "PlxChipFn.h"
#include "PlxInterrupt.h"
#include "SuppFunc.h"




/******************************************************************************
 *
 * Function   :  PlxChipInterruptsEnable
 *
 * Description:  Globally enables PLX chip interrupts
 *
 *****************************************************************************/
BOOLEAN
PlxChipInterruptsEnable(
    DEVICE_EXTENSION *pdx
    )
{
    U32 RegInterrupt;


    // Enable PCI interrupt
    RegInterrupt =
        PLX_9000_REG_READ(
            pdx,
            PCI9056_INT_CTRL_STAT
            );

    PLX_9000_REG_WRITE(
        pdx,
        PCI9056_INT_CTRL_STAT,
        RegInterrupt | (1 << 8)
        );

    return TRUE;
}




/******************************************************************************
 *
 * Function   :  PlxChipInterruptsDisable
 *
 * Description:  Globally disables PLX chip interrupts
 *
 *****************************************************************************/
BOOLEAN
PlxChipInterruptsDisable(
    DEVICE_EXTENSION *pdx
    )
{
    U32 RegInterrupt;


    // Disable PCI interrupt
    RegInterrupt =
        PLX_9000_REG_READ(
            pdx,
            PCI9056_INT_CTRL_STAT
            );

    PLX_9000_REG_WRITE(
        pdx,
        PCI9056_INT_CTRL_STAT,
        RegInterrupt & ~(1 << 8)
        );

    return TRUE;
}




/******************************************************************************
 *
 * Function   :  PlxChipSetInterruptNotifyFlags
 *
 * Description:  Sets the interrupt notification flags of a wait object
 *
 ******************************************************************************/
VOID
PlxChipSetInterruptNotifyFlags(
    PLX_INTERRUPT   *pPlxIntr,
    PLX_WAIT_OBJECT *pWaitObject
    )
{
    // Clear notify events
    pWaitObject->Notify_Flags = INTR_TYPE_NONE;

    if (pPlxIntr->PciAbort)
        pWaitObject->Notify_Flags |= INTR_TYPE_PCI_ABORT;

    if (pPlxIntr->LocalToPci & (1 << 0))
        pWaitObject->Notify_Flags |= INTR_TYPE_LOCAL_1;

    if (pPlxIntr->DmaDone & (1 << 0))
        pWaitObject->Notify_Flags |= INTR_TYPE_DMA_0;

    if (pPlxIntr->DmaDone & (1 << 1))
        pWaitObject->Notify_Flags |= INTR_TYPE_DMA_1;

    if (pPlxIntr->MuOutboundPost)
        pWaitObject->Notify_Flags |= INTR_TYPE_OUTBOUND_POST;

    pWaitObject->Notify_Doorbell = pPlxIntr->Doorbell;
}




/******************************************************************************
 *
 * Function   :  PlxChipSetInterruptStatusFlags
 *
 * Description:  Sets the interrupts that triggered notification
 *
 ******************************************************************************/
VOID
PlxChipSetInterruptStatusFlags(
    PLX_INTERRUPT_DATA *pIntData,
    PLX_INTERRUPT      *pPlxIntr
    )
{
    // Clear all interrupt flags
    RtlZeroMemory(
        pPlxIntr,
        sizeof(PLX_INTERRUPT)
        );

    if (pIntData->Source_Ints & INTR_TYPE_PCI_ABORT)
        pPlxIntr->PciAbort = 1;

    if (pIntData->Source_Ints & INTR_TYPE_LOCAL_1)
        pPlxIntr->LocalToPci = (1 << 0);

    if (pIntData->Source_Ints & INTR_TYPE_DMA_0) 
        pPlxIntr->DmaDone |= (1 << 0);

    if (pIntData->Source_Ints & INTR_TYPE_DMA_1) 
        pPlxIntr->DmaDone |= (1 << 1);

    if (pIntData->Source_Ints & INTR_TYPE_OUTBOUND_POST)
        pPlxIntr->MuOutboundPost = 1;

    pPlxIntr->Doorbell = pIntData->Source_Doorbell;
}




/*******************************************************************************
 *
 * Function   :  PlxChipTypeDetect
 *
 * Description:  Attempts to determine PLX chip type and revision
 *
 ******************************************************************************/
PLX_STATUS
PlxChipTypeDetect(
    DEVICE_EXTENSION *pdx
    )
{
    U32 RegValue;


    // Set default values
    pdx->Key.PlxChip     = PLX_CHIP_TYPE;
    pdx->Key.PlxRevision = pdx->Key.Revision;
    pdx->Key.PlxFamily   = PLX_FAMILY_BRIDGE_P2L;

    // Check hard-coded ID
    RegValue =
        PLX_9000_REG_READ(
            pdx,
            0x70
            );

    if ((RegValue & 0xFFFF) == PLX_VENDOR_ID)
    {
        pdx->Key.PlxChip = (U16)(RegValue >> 16);

        // Get revision
        RegValue =
            PLX_9000_REG_READ(
                pdx,
                0x74           // Revision ID
                );

        pdx->Key.PlxRevision = (U8)RegValue;
    }

    DebugPrintf((
        "Device %04X_%04X = %04X rev %02X\n",
        pdx->Key.DeviceId, pdx->Key.VendorId,
        pdx->Key.PlxChip, pdx->Key.PlxRevision
        ));

    return ApiSuccess;
}




/******************************************************************************
 *
 * Function   :  PlxChipGetRemapOffset
 *
 * Description:  Returns the remap register offset for a PCI BAR space
 *
 ******************************************************************************/
VOID
PlxChipGetRemapOffset(
    DEVICE_EXTENSION *pdx,
    U8                BarIndex,
    U16              *pOffset_RegRemap
    )
{
    U32     RegValue;
    BOOLEAN bBarsShifted;


    // Check if BAR2/BAR3 are shifted to BAR0/BAR1
    RegValue =
        PLX_9000_REG_READ(
            pdx,
            PCI9056_ENDIAN_DESC
            );

    if ((RegValue & 0x300) == 0x200)
    {
        bBarsShifted = TRUE;
    }
    else
    {
        bBarsShifted = FALSE;
    }

    switch (BarIndex)
    {
        case 0:
            /**************************************************
             * Space 1 is a special case.  If the I2O Decode
             * enable bit is set, BAR3 is moved to BAR0
             *************************************************/

            // Check if I2O decode is enabled
            RegValue =
                PLX_9000_REG_READ(
                    pdx,
                    PCI9056_FIFO_CTRL_STAT
                    );

            if (RegValue & (1 << 0))
            {
                // I2O Decode is enbled, use BAR0 for Space 1
                *pOffset_RegRemap = PCI9056_SPACE1_REMAP;
                return;
            }
            break;

        case 1:
            // BAR 1 could be Space 0 if shifted
            if (bBarsShifted)
            {
                *pOffset_RegRemap = PCI9056_SPACE0_REMAP;
                return;
            }
            break;
            
        case 2:
            // BAR 2 could be Space 0 or Space 1 if shifted
            if (bBarsShifted)
                *pOffset_RegRemap = PCI9056_SPACE1_REMAP;
            else
                *pOffset_RegRemap = PCI9056_SPACE0_REMAP;
            return;

        case 3:
            // BAR 3 can only be Space 1
            *pOffset_RegRemap = PCI9056_SPACE1_REMAP;
            return;
    }

    DebugPrintf(("ERROR - Invalid Space\n"));

    // BAR not supported
    *pOffset_RegRemap = (U16)-1;
}




/******************************************************************************
 *
 * Function   :  PlxChipPowerContextSave
 *
 * Description:  Save device-specific information when the device goes to low power state
 *
 *****************************************************************************/
BOOLEAN
PlxChipPowerContextSave(
    DEVICE_EXTENSION *pdx
    )
{
    U16 i;
    U16 NumRegs;


    if (pdx->pSavedContext != NULL)
    {
        DebugPrintf(("ERROR - Context save buffer already exists\n"));
        return FALSE;
    }

    // Calculate number of registers to save (PCI + local)
    NumRegs = 16 + PLX_CHIP_CONTEXT_TOTAL_REGS;

    // Allocate buffer to store registers
    pdx->pSavedContext =
        ExAllocatePoolWithTag(
            NonPagedPool,
            NumRegs * sizeof(U32),
            (ULONG)' XLP'       // "PLX " in reverse order
            );

    if (pdx->pSavedContext == NULL)
    {
        DebugPrintf(("ERROR - Unable to allocate context buffer\n"));
        return FALSE;
    }

    // Save device context information
    for (i=0; i < NumRegs; i++)
    {
        if (i < 16)
        {
            // Save PCI register
            PLX_PCI_REG_READ(
                pdx,
                i * sizeof(U32),
                &pdx->pSavedContext[i]
                );
        }
        else
        {
            // Save local register
            pdx->pSavedContext[i] =
                PLX_9000_REG_READ(
                    pdx,
                    (i - 16) * sizeof(U32)
                    );
        }
    }

    return TRUE;
}




/******************************************************************************
 *
 * Function   :  PlxChipPowerContextRestore
 *
 * Description:  Restore device-specific information when the device goes to full power state
 *
 *****************************************************************************/
BOOLEAN
PlxChipPowerContextRestore(
    DEVICE_EXTENSION *pdx
    )
{
    U16     i;
    U16     NumRegs;
    U32     RegValue;
    BOOLEAN bRestorePci;


    if (pdx->pSavedContext == NULL)
    {
        DebugPrintf(("ERROR - Context information has not been saved\n"));
        return FALSE;
    }

    // A return to D0 state should cause a local reset. A manual reset is forced
    // to ensure the EEPROM gets re-loaded.
    PlxPciDeviceReset(
        pdx
        );

    // Get PCI BAR 0
    PLX_PCI_REG_READ(
        pdx,
        0x10,
        &RegValue
        );

    // If BAR 0 assigned valid address, don't restore PCI registers
    if ((RegValue & ~0xF) == 0)
    {
        DebugPrintf(("-- PCI regs not reset, implement workaround to restore them\n"));
        bRestorePci = TRUE;
    }
    else
    {
        bRestorePci = FALSE;
    }

    // Calculate number of registers to restore (PCI + local)
    NumRegs = 16 + PLX_CHIP_CONTEXT_TOTAL_REGS;

    // Restore device context information
    for (i=0; i < NumRegs; i++)
    {
        if (i < 16)
        {
            if (bRestorePci)
            {
                // Restore PCI register
                PLX_PCI_REG_WRITE(
                    pdx,
                    i * sizeof(U32),
                    pdx->pSavedContext[i]
                    );
            }
        }
        else
        {
            // Restore local register
            PLX_9000_REG_WRITE(
                pdx,
                (i - 16) * sizeof(U32),
                pdx->pSavedContext[i]
                );
        }
    }

    // Release context save buffer
    ExFreePool(
        pdx->pSavedContext
        );

    // Mark buffer as released
    pdx->pSavedContext = NULL;

    return TRUE;
}
