#ifndef __DRIVER_DEFS_H
#define __DRIVER_DEFS_H

/*******************************************************************************
 * Copyright (c) PLX Technology, Inc.
 *
 * PLX Technology Inc. licenses this source file under the GNU Lesser General Public
 * License (LGPL) version 2.  This source file may be modified or redistributed
 * under the terms of the LGPL and without express permission from PLX Technology.
 *
 * PLX Technology, Inc. provides this software AS IS, WITHOUT ANY WARRANTY,
 * EXPRESS OR IMPLIED, INCLUDING, WITHOUT LIMITATION, ANY WARRANTY OF
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE.  PLX makes no guarantee
 * or representations regarding the use of, or the results of the use of,
 * the software and documentation in terms of correctness, accuracy,
 * reliability, currentness, or otherwise; and you rely on the software,
 * documentation and results solely at your own risk.
 *
 * IN NO EVENT SHALL PLX BE LIABLE FOR ANY LOSS OF USE, LOSS OF BUSINESS,
 * LOSS OF PROFITS, INDIRECT, INCIDENTAL, SPECIAL OR CONSEQUENTIAL DAMAGES
 * OF ANY KIND.
 *
 ******************************************************************************/

/******************************************************************************
 *
 * File Name:
 *
 *      DrvDefs.h
 *
 * Description:
 *
 *      Common definitions used in the driver
 *
 * Revision History:
 *
 *      05-01-13 : PLX SDK v7.10
 *
 ******************************************************************************/


#include "Plx.h"
#include "PlxTypes.h"




/**********************************************
 *               Definitions
 *********************************************/
#define PLX_DRIVER_NAME                     "PlxSvc"
#define PLX_DRIVER_NAME_UNICODE             L"PlxSvc"
#define PLX_MAX_NAME_LENGTH                 0x20          // Max length of registered device name
#define DEFAULT_SIZE_COMMON_BUFFER          (8 * 1024)    // Default size of Common Buffer


// Set log destination
#if defined(PLX_LOG_TO_FILE)
    #define PLX_DBG_DEST_FILE
#else
    #define PLX_DBG_DEST_DEBUGGER
#endif


// Macros to support Kernel-level logging in Debug builds
#if defined(PLX_DBG_DEST_DEBUGGER)
    #define _PlxDbgFunc                     DbgPrint
#elif defined(PLX_DBG_DEST_FILE)
    #define _PlxDbgFunc                     PlxLogPrintf
    #define PLX_LOG_FILE_UNICODE            L"\\SystemRoot\\PlxSvc.log"
#endif

#if defined(PLX_DEBUG)
    #define DebugPrintf(arg)                _Debug_Print_Macro(arg)
    #define DebugPrintf_Cont(arg)           _PlxDbgFunc arg
#else
    #define DebugPrintf(arg)                do { } while(0)
    #define DebugPrintf_Cont(arg)           do { } while(0)
#endif
#define ErrorPrintf(arg)                    _Debug_Print_Macro(arg)
#define ErrorPrintf_Cont(arg)               _PlxDbgFunc arg

#define _Debug_Print_Macro(arg)            \
    do                                     \
    {                                      \
        _PlxDbgFunc(PLX_DRIVER_NAME ": "); \
        _PlxDbgFunc arg;                   \
    }                                      \
    while (0)



// Macros for I/O port access
#define IO_PORT_READ_8(port)                        READ_PORT_UCHAR ( (U8*) (port) )
#define IO_PORT_READ_16(port)                       READ_PORT_USHORT( (U16*)(port) )
#define IO_PORT_READ_32(port)                       READ_PORT_ULONG ( (U32*)(port) )
#define IO_PORT_WRITE_8( port, value)               WRITE_PORT_UCHAR ( (U8*) (port), (value) )
#define IO_PORT_WRITE_16(port, value)               WRITE_PORT_USHORT( (U16*)(port), (value) )
#define IO_PORT_WRITE_32(port, value)               WRITE_PORT_ULONG ( (U32*)(port), (value) )


// Macros for memory access
#define PHYS_MEM_READ_8(addr)                       READ_REGISTER_UCHAR  ( (U8*) (addr) )
#define PHYS_MEM_READ_16(addr)                      READ_REGISTER_USHORT ( (U16*)(addr) )
#define PHYS_MEM_READ_32(addr)                      READ_REGISTER_ULONG  ( (U32*)(addr) )
#define PHYS_MEM_WRITE_8( addr, value)              WRITE_REGISTER_UCHAR  ( (U8*) (addr), (value) )
#define PHYS_MEM_WRITE_16(addr, value)              WRITE_REGISTER_USHORT ( (U16*)(addr), (value) )
#define PHYS_MEM_WRITE_32(addr, value)              WRITE_REGISTER_ULONG  ( (U32*)(addr), (value) )

// 64-bit macros not provided by WDK for 32-bit and only Vista WDK or higher
#if !defined(PLX_64BIT) || (WINVER <= 0x0502)
    #define READ_REGISTER_ULONG64(reg)              (*(U64*)(reg))
    #define WRITE_REGISTER_ULONG64(reg, value)      (*(U64*)(reg) = (value))
#endif



// Macros for PLX chip register access
#define PLX_8111_REG_READ(pNode, offset)            PlxRegisterRead_8111 ((pNode), (offset), NULL)
#define PLX_8111_REG_WRITE(pNode, offset, val)      PlxRegisterWrite_8111((pNode), (offset), (val))

#define PLX_8000_REG_READ(pNode, offset)            PlxRegisterRead_8000 ((pNode), (offset), NULL, FALSE)
#define PLX_8000_REG_WRITE(pNode, offset, val)      PlxRegisterWrite_8000((pNode), (offset), (val), FALSE)




/***********************************************************
 * The following definition sets the maximum size that an
 * MDL can describe.  Although not documented in the DDK
 * until recently, the memory allocated for an MDL is 64k,
 * which limits the number of physical page entries allowed
 * after the MDL.
 *
 * Assuming 4k page sizes, the limit calculation yields a
 * size of just under 64MB on a 32-bit system.  On a system with
 * 64-bit addressing, the max size per MDL is less than 32MB.
 * The calculation for determining the max size is:
 *
 *   Total Mem = PAGE_SIZE * ((64k - sizeof(MDL)) / sizeof(PLX_UINT_PTR))
 *
 * The total size is decremented by an additional amount because
 * Windows may still fail mappings in certain cases if the size
 * MAX_MDL_SIZE is too large.
 **********************************************************/
#define MAX_MDL_SIZE                        (PAGE_SIZE * ((65535 - sizeof(MDL)) / sizeof(PLX_UINT_PTR))) - (1 << 21)



/***********************************************************
 * The following definition sets the maximum number of
 * consecutive virtual addresses for a PCI BAR space.  Since
 * individual MDLs are limited to 64MB, when mapping a space
 * to obtain a user-mode virtual address, each individual
 * MDL for a space must be mapped separately.  The virtual
 * addresses must be saved for later unmapping.
 *
 * The following calculation determines the theoretical
 * limitation of a user mapping for a PCI space:
 *
 *   Max mapping size = (Max # virt addresses) * (Max MDL size)
 * 
 * If the max virtual address count is 5, for example, and
 * each MDL is limited to 60MB, the theoretical limit that the
 * driver can map will be a 300MB PCI space.
 *
 * NOTE: Increasing this number has no effect if a mapping
 *       fails due to system resource constraints.  It is
 *       primarily used to store multiple addresses so they
 *       can later be unmapped properly.
 **********************************************************/
#define MAX_VIRTUAL_ADDR                    10      // Max number of virtual addresses for a space



// Information stored in Registry
typedef struct _PLX_REGISTRY_INFO
{
    U32 Size_CommonBuffer;
} PLX_REGISTRY_INFO;


// Information about contiguous, page-locked buffers
typedef struct _PLX_PHYS_MEM_OBJECT
{
    LIST_ENTRY  ListEntry;
    VOID       *pOwner;
    U8         *pKernelVa;
    U64         CpuPhysical;                    // CPU Physical Address
    U64         BusPhysical;                    // Bus Physical Address
    U32         Size;                           // Buffer size
    PMDL        pMdl;
    BOOLEAN     bCacheable;
    LIST_ENTRY  List_Mappings;                  // List of mappings for this physical memory
    KSPIN_LOCK  Lock_MappingsList;              // Spinlock for mappings list
} PLX_PHYS_MEM_OBJECT;


// Mappings to user virtual space
typedef struct _PLX_USER_MAPPING
{
    LIST_ENTRY  ListEntry;
    VOID       *pOwner;
    U8          BarIndex;
    VOID       *pUserVa[MAX_VIRTUAL_ADDR];
} PLX_USER_MAPPING;


// PCI BAR Space information
typedef struct _PLX_PCI_BAR_INFO
{
    U8               *pVa;                      // BAR Kernel Virtual Address
    PLX_PCI_BAR_PROP  Properties;               // BAR Properties
    PMDL              pMdl;                     // MDL for the BAR space
} PLX_PCI_BAR_INFO;


// Device node information
typedef struct _PLX_DEVICE_NODE
{
    LIST_ENTRY                ListEntry;
    struct _DEVICE_EXTENSION *pdx;                          // Pointer to parent device object
    struct _PLX_DEVICE_NODE  *pParent;                      // The parent P2P port
    struct _PLX_DEVICE_NODE  *pRegNode;                     // The device to use for register access
    PLX_DEVICE_KEY            Key;                          // Device location & identification
    U8                        PciHeaderType;                // PCI header type
    U32                       PciClass;                     // PCI class code
    PLX_PCI_BAR_INFO          PciBar[PCI_NUM_BARS_TYPE_00]; // PCI BARs information
    U8                        bBarKernelMap;                // Flag whether BARs have been mapped to kernel
    U8                        PortNumber;                   // PCIe Port number of device
    U8                        Default_EepWidth;             // Default width for EEPROM access
    U8                        nMappedCommonBuffer;          // Number of mappings to common buffer
    U32                       Offset_NtRegBase;             // The NT port register base offset
    LIST_ENTRY                List_BarMappings;             // List of mappings to user space
    KSPIN_LOCK                Lock_BarMappingsList;         // Spinlock for user mappings list
    LIST_ENTRY                List_PhysicalMem;             // List of user-allocated physical memory
    KSPIN_LOCK                Lock_PhysicalMemList;         // Spinlock for physical memory list
} PLX_DEVICE_NODE;


// All relevant information about the device
typedef struct _DEVICE_EXTENSION
{
    DEVICE_OBJECT  *pDeviceObject;                  // Device object this extension belongs to
    WCHAR           DriverName[PLX_MAX_NAME_LENGTH];// Registered driver name
    WCHAR           LinkName[PLX_MAX_NAME_LENGTH];  // Registered symbolic link name
    U8              OpenCount;                      // Count of active open descriptors
    LIST_ENTRY      List_Devices;                   // List of detected devices
} DEVICE_EXTENSION;



#endif
